# !/bin/bash
# Daemon waits for an Update Trigger, and
# runs the update process when it gets one.
#
# Copyright (c) RAYLASE GmbH 2015-2025
#

VERSION="V1.1.9"

# Variable configuration and function definitions
. upd_functions.sh

# Error code definitions
. upd_err.sh

SPICE3_DIR="/opt/RAYLASE/SPICE3"
POWERLEDPROG="${SPICE3_DIR}/bin/powerLed"

UPDATEPROGRESS_FILE="${SPICE3_DIR}/lib/Content/update.txt"

UBOOT_IMAGE_NAME="u-boot.img"
UBOOT_SCRIPT_NAME="u-boot.scr"
UBOOT_ENV_SCRIPT_NAME="u-boot-env.scr"
UBOOT_BOOTPART_COPY="bootpart.copy"

#
# Construct an error message from the symbolic errorcode name,
# then tack on any other supplied args.
# Send the message to the update log file, and to the
# update progress text file, which the WebIF reads periodically.
#
output_err_msg() {
	local errname="$1"

	# insert the numeric value of the errorcode, followed by the associated error text.
	local errmsg="${!errname}: ${UPD_ERR_MSG[${errname}]}"

	# see if there is anything else to add to the message.
	if (($# > 1)) ; then
		shift
		errmsg="${errmsg}: $*"
	fi
	logecho "Error: ${errmsg}"
	echo "${errmsg}" > "$UPDATEPROGRESS_FILE"
}

init_daemon() {

	logecho "Initialising Update Daemon $VERSION for SP-ICE-3."

	if [ "$(whoami)" != "root" ] ; then
		fatal "Must be root."
	fi

	mountall

	# Read U-Boot logs, and add them to the update log
	# So that all log entries are in correct sequence
	shopt -s nullglob
	if [[ -f "${FAT_DIR}/${UBOOT_BOOTPART_COPY}" ]]; then
		logecho "U-BOOT start status copy (${UBOOT_BOOTPART_COPY}): $(< ${FAT_DIR}/${UBOOT_BOOTPART_COPY} )"
		# Replace flags in bootpart.copy without deleting/recreating or changing the size of the file
		printf '\0\0\0\0' | dd of="${FAT_DIR}/${UBOOT_BOOTPART_COPY}" bs=1 seek=0 count=4 status=none
	fi
	for myfile in ${FAT_DIR}/ukn*txt ${FAT_DIR}/kg*txt ${FAT_DIR}/okg*txt ; do
		UB_STATALL=$(< "${myfile}")
		rm "${myfile}"
		UB_TRYBOOT="${UB_STATALL:0:1}"
		UB_UNKNOWN_NEW="${UB_STATALL:1:1}"
		UB_KNOWN_GOOD="${UB_STATALL:2:1}"
		UB_OLD_KNOWN_GOOD="${UB_STATALL:3:1}"
		logecho "U-BOOT end status (${myfile##*/}): try_boot=${UB_TRYBOOT}, unknown_new=${UB_UNKNOWN_NEW}, known_good=${UB_KNOWN_GOOD} old_known_good=${UB_OLD_KNOWN_GOOD}"
	done

	# Check & log which image is running currently
	checkbootedimage

	# Refresh the current status file.
	refresh_status

	DAEMON_PID=$$
	logecho "PID of Update Daemon = ${DAEMON_PID}"
}


# Umount all partitions expect (presumeably) the one we're running from...
umountall_except() {
	local CURR_PART=$1
	echo "Unmounting all partitions for reboot."
	for part in $FAT_PART $IMG0_PART $IMG1_PART $IMG2_PART $UPD_PART ; do
		if (( CURR_PART != part )); then
			umount ${MMCDEVICE}${part} || echo "Warning: couldn't umount Partition ${part} for reboot."
		fi
	done
}

# Unpack tarball to destination directory
# $1 package name to unpack
# $2 destination directory
unpack_update() {

	echo 15 > "$UPDATEPROGRESS_FILE"

	if [ -z "$4" ] ; then
		output_err_msg 'UPD_ERR_NOIMGNUM'
		return 1
	fi
	if [ -z "$3" ] ; then
		output_err_msg 'UPD_ERR_NOIMGPART'
		return 1
	fi
	if [ -z "$2" ] ; then
		output_err_msg 'UPD_ERR_NODEST'
		return 1
	fi
	if [ -z "$1" ] ; then
		output_err_msg 'UPD_ERR_NOPKGNAME'
		return 1
	fi
	
	if [ ! -f "$1" ] ; then
		output_err_msg 'UPD_ERR_PKGNOTEXIST' "$1"
		return 1
	fi

	local updpkg="$1"
	local NEW_IMAGE_PART="$3"
	local NEW_IMAGE_DIR="$2"
	
	logecho "Unmounting ${MMCDEVICE}${NEW_IMAGE_PART} from ${NEW_IMAGE_DIR}"
	umount "${MMCDEVICE}${NEW_IMAGE_PART}"
	
	# u-boot v2015 does not understand 64-bit ext4.
	local mkfsflags="-q -F -F -O ^64bit -O ^orphan_file "
	
	logecho "Formatting partition ${MMCDEVICE}${NEW_IMAGE_PART} as 'SYS${4}'"
	mkfs.ext4 ${mkfsflags} -L "SYS${4}" "${MMCDEVICE}${NEW_IMAGE_PART}"
	
	logecho "Re-Mounting ${MMCDEVICE}${NEW_IMAGE_PART} on ${NEW_IMAGE_DIR}"
	mount "${MMCDEVICE}${NEW_IMAGE_PART}" "${NEW_IMAGE_DIR}"
	echo 25 > $UPDATEPROGRESS_FILE
	logecho "Extracting contents of '$1' to ${NEW_IMAGE_DIR}"
	if [[ $updpkg == *".tar.bz2" ]]; then
		tar -xj --totals -f "${updpkg}" -C "${NEW_IMAGE_DIR}" >> "${LOG_FILE}" 2>&1
	elif [[ $updpkg == *".tar.xz" ]]; then
		tar -xJ --totals -f "${updpkg}" -C "${NEW_IMAGE_DIR}" >> "${LOG_FILE}" 2>&1
	fi
	if [ $? != 0 ] ; then
		output_err_msg 'UPD_ERR_PKGEXTRFAIL' "${updpkg}"
		return 1
	fi
	echo 60 > $UPDATEPROGRESS_FILE
	logecho "Extraction finished successfully"
	return 0;
}

delete_used_package() {
	# Regardless of whether extraction of the contents succeeded or failed,
	# we have to delete the package file in order to avoid endless looping
	logecho "Deleting used package '${updpkg}'"
	rm -f ${updpkg}
}

# Perform update
do_update() {
	local updpkg="$1"
	logecho "Update procedure started: using '${updpkg}'"
	echo 10 > $UPDATEPROGRESS_FILE
	${POWERLEDPROG} 3

	#
	# now see what we're dealing with: a normal update, or a components package
	#
	case "${updpkg##*/}" in 

		"components"* )
			#
			# Extract everything from the components tarball.
			#
			# If an update-package is extracted, it will be processed later.
			#
			logecho "Extracting contents of '${updpkg}'"
			tar -xj --totals -f "${updpkg}" -C "${UPD_DIR}" >> "${LOG_FILE}" 2>&1
			if [ $? != 0 ] ; then
				output_err_msg 'UPD_ERR_PKGEXTRFAIL' "${updpkg}"
				delete_used_package "${updpkg}"
				return 1
			fi

			#
			# Process U-BOOT bits and pieces
			#
			logecho "Copying u-boot stuff..."
			for f in "${UBOOT_IMAGE_NAME}" "${UBOOT_SCRIPT_NAME}" "${UBOOT_ENV_SCRIPT_NAME}" ; do
				if [ -f "${UPD_DIR}/${f}" ]; then
					cp -v -f --backup=simple --suffix=".bak" "${UPD_DIR}/${f}" "${FATDIR}/${f}" >> "${LOG_FILE}" 2>&1
					if [ $? != 0 ] ; then
						output_err_msg 'UPD_ERR_CPYTOFATFAIL' "'${UPD_DIR}/${f}' to '${FATDIR}/${f}'"
						delete_used_package "${updpkg}"
						return 1
					fi
					rm -v -f "${UPD_DIR}/${f}"
				fi
			done

			#
			# That's all we do for the components package.
			#
			delete_used_package "${updpkg}"
			return 0
		;;

	esac	



	local NEW_IMAGE='1'
	local NEW_IMAGE_DIR="${IMG1_DIR}"
	local NEW_IMAGE_PART="${IMG1_PART}"
	
	if [ "${BOOTED_IMAGE}" == '1' ] ; then
		NEW_IMAGE='2'
		NEW_IMAGE_DIR="${IMG2_DIR}"
		NEW_IMAGE_PART="${IMG2_PART}"
	elif [ "${BOOTED_IMAGE}" == '2' ] ; then
		NEW_IMAGE='1'
		NEW_IMAGE_DIR="${IMG1_DIR}"
		NEW_IMAGE_PART="${IMG1_PART}"
	elif [ "${BOOTED_IMAGE}" == '0' ] ; then
		# Only, if both images #1 and #2 were bad, we will start image #0
		NEW_IMAGE='1'
		NEW_IMAGE_DIR="${IMG1_DIR}"
		NEW_IMAGE_PART="${IMG1_PART}"
	fi

	logecho "Currently running image #${BOOTED_IMAGE}: install new image as #${NEW_IMAGE} in ${NEW_IMAGE_DIR}"
	
	unpack_update "${updpkg}" "${NEW_IMAGE_DIR}" "${NEW_IMAGE_PART}" "${NEW_IMAGE}"
	local unpack_data_return_code=$?
	
	# Regardless of whether update_unpack succeeded or failed,
	# we have to delete the update package file in order to avoid endless looping
	delete_used_package "${updpkg}"
	
	if [ ${unpack_data_return_code} == 0 ] ; then
		local unmounted="false"
		logecho "Unmounting ${NEW_IMAGE_DIR} before reboot"
		for (( i=0; i < 10; ++i )) ; do
			if sudo umount "${MMCDEVICE}${NEW_IMAGE_PART}" &&	sudo e2fsck -f -n "${MMCDEVICE}${NEW_IMAGE_PART}" ; then
			
				unmounted="true"
				break
				
			fi
			logecho "Retrying unmount $i"
			sleep 1
		done
		echo 66 > $UPDATEPROGRESS_FILE
		if [ "${unmounted}" == "true" ] ; then
			logecho "Unmounting ${NEW_IMAGE_DIR} succeeded"

			IMG_UNKNOWN_NEW="$NEW_IMAGE"
			logecho "Refreshing boot status after update"
			write_status_variables_bin
			read_status_variables_bin
			
			logecho "---- UPDATE FINISHED: REBOOTING NOW -----"

			#
			# quasi rotate the update log (if necessary), keeping a max of 10 old logs
			#
			mv "${LOG_FILE}" "${LOG_FILE%%.log}.$(date +%Y-%m-%d.%H%M%S).log"
			ls -1tr "${LOG_FILE%%.log}"*".log" | head -n -10 | xargs -d '\n' rm -f

			sync
			sleep 2
			umountall_except "${BOOTED_IMAGE}"
			echo "68" > "$UPDATEPROGRESS_FILE"
			reboot
		else
			output_err_msg 'UPD_ERR_UMNTIMGDIRFAIL' "'${NEW_IMAGE_DIR}'"
		fi
	fi
}

start_update() {
	echo "1" > "$UPDATEPROGRESS_FILE"
	logecho "SIGHUP received, looking for an update package"
	local foundPackages=0
	#
	# check if there is at least one components and/or update package file available...
	#
	if [ -n "$(find ${UPD_DIR} -maxdepth 1 -name 'components*.tar.*' -print -quit)" ]; then
		echo "5" > "$UPDATEPROGRESS_FILE"
		for updpkg in `ls ${UPD_DIR}/components*.tar.* 2>/dev/null` ; do
			logecho "Found components package: '${updpkg}'"
			(( ++foundPackages ))
			# Give some time to relax
			sync
			sleep 2
			sync
			do_update "$updpkg"
		done
	fi
	if [ -n "$(find ${UPD_DIR} -maxdepth 1 -name 'update*.tar.*' -print -quit)" ]; then
		echo "5" > "$UPDATEPROGRESS_FILE"
		for updpkg in `ls ${UPD_DIR}/update*.tar.* 2>/dev/null` ; do
			logecho "Found update package: '${updpkg}'"
			(( ++foundPackages ))
			# Give some time to relax
			sync
			sleep 2
			sync
			do_update "$updpkg"
		done
	fi
	if (( foundPackages < 1 )); then
		output_err_msg 'UPD_ERR_NOPKGINDIR' "'${UPD_DIR}'"
	fi
	sleep 1
}

# -----------------------------------------------------------------
#
# main function
#

init_daemon

logecho "Update Daemon $VERSION for SP-ICE-3 started."
echo "0" > "$UPDATEPROGRESS_FILE"
chmod 666 $UPDATEPROGRESS_FILE

# Start an update, after receiving the signal
trap start_update SIGHUP

while :
do
	# Simply do nothing, but do it interuptably (so we can service signals immediately).
	sleep 120 &
	wait $!
done

