# !/bin/bash
# Commonly used functions for starter and daemon
#
# (c) RAYLASE GmbH 2015

# This is the file where the boot status is kept on the UBOOT (FAT) partition.
export BOOTPART="bootpart"

# Device node for MMC-Device
MMCDEVNAME="mmcblk0p"
MMCDEVICE="/dev/${MMCDEVNAME}"

# Temporary directory in ramdisk
TEMPDIR="/tmp"

# Partition numbers for partitions
FAT_PART=1
#SPL_PART=2
IMG0_PART=3
#EXT_PART=4
IMG1_PART=5
IMG2_PART=6
UPD_PART=7
DATA_PART=8

# Mount points
FAT_DIR="/media/fat"
HOME_DIR="/home"
IMG0_DIR="/media/image0"
IMG1_DIR="/media/image1"
IMG2_DIR="/media/image2"
UPD_DIR="/var/updates"
# Not relevant for update
# DATA_DIR=/data

MYDIR="/home/root/update_spice3"

LOG_DIR="${HOME_DIR}/spice3/log"
OLDLOG_DIR="${FAT_DIR}/log"
LOG_FILE="${LOG_DIR}/update.spice3.log"

# Status variables, maintained in statfile
export STAT_TRY_BOOT='X'
export IMG_UNKNOWN_NEW='X'
export IMG_KNOWN_GOOD='X'
export IMG_OLD_KNOWN_GOOD='X'

# Number of currently booted image
export BOOTED_IMAGE='0'

logecho_init() {
	# Mount home Partition in order to store the logfile there
	if [ ! -d "${HOME_DIR}" ] ; then
		mkdir -p "${HOME_DIR}" || fatal "Could not create mount point $HOME_DIR"
	fi
	# mount, if not already done
	mountpoint -q "${HOME_DIR}" || mount "${MMCDEVICE}${DATA_PART}" "${HOME_DIR}" -o rw || fatal "Could not mount $HOME_DIR"
	# make sure we have a log sub-dir
	mkdir -p "${LOG_DIR}"
	# Everybody is allowed to write in the LOG directory
	chmod 777 "${LOG_DIR}"
	
	if [ ! -d "${FAT_DIR}" ] ; then
		mkdir -p "${FAT_DIR}" || fatal "Could not create mount point $FAT_DIR"
	fi
	mountpoint -q "${FAT_DIR}" || mount "${MMCDEVICE}${FAT_PART}" "${FAT_DIR}" -o rw,umask=0000 || fatal "Could not mount $FAT_DIR"
	
	if [ -d "${OLDLOG_DIR}" ] ; then
		if [ ! -z "$(ls -A "$OLDLOG_DIR")" ]; then
			mv "${OLDLOG_DIR}"/* "${LOG_DIR}"
			if [ ! -z "$(ls -A "$OLDLOG_DIR")" ]; then
				logecho "Moving all logfiles from the old logging directory failed!"
			fi
		fi
		rmdir "${OLDLOG_DIR}"
		# Fix permissions since fat does not know Unix permissions
		find "${LOG_DIR}" -type f -exec chmod 666 {} +
	fi

	logecho "--------- Update Log Started --------"
}

logecho() {
	echo "$(date '+%Y-%m-%dT%H:%M:%S.%3N') $*" >>"${LOG_FILE}"
}

# Exit on fatal errors
fatal() {
	logecho "************ Fatal error ************"
	logecho $*
	logecho "******** Terminating program ********"
	exit 1
}

# Check which image is running currently
checkbootedimage ()
{
	local BOOTEDPART=$(cat /proc/cmdline | grep -o "${MMCDEVICE}[${IMG0_PART}${IMG1_PART}${IMG2_PART}]")
	if [ $? == 0 ] ; then
		BOOTEDPARTNUM="${BOOTEDPART#${MMCDEVICE}}"
		BOOTED_IMAGE=0
		case "${BOOTEDPARTNUM}" in 
			"${IMG1_PART}" )
				BOOTED_IMAGE=1
				;;
			"${IMG2_PART}" )
				BOOTED_IMAGE=2
				;;
		esac
		local BOOTEDPARTLABEL=$(ls -l /dev/disk/by-label | grep "${MMCDEVNAME}${BOOTEDPARTNUM}" | awk '{print $9}')
		
		logecho "Booted from Image ${BOOTED_IMAGE}, label ${BOOTEDPARTLABEL}, partition ${BOOTEDPART}"
		logecho "SP-ICE-3 Version: $(tr '\n' ' ' < /spice3-image-version )"
	else
		fatal "Not able to detect boot image number"
	fi
}

# Mount required directories
# Note: $HOME_DIR is mounted already earlier for logging
mountall() {
#	# TODO: Define Permissions and ownership for directories
	for dirname in "$FAT_DIR" "$IMG0_DIR" "$IMG1_DIR" "$IMG2_DIR" "$UPD_DIR" ; do
		chmod 777 "$dirname"
	done
}

# Store status variables
write_status_variables_bin() {
	logecho "Writing status: try_boot=${STAT_TRY_BOOT}, unknown_new=${IMG_UNKNOWN_NEW}, known_good=${IMG_KNOWN_GOOD}, old_known_good=${IMG_OLD_KNOWN_GOOD}"
	echo -n "${STAT_TRY_BOOT}${IMG_UNKNOWN_NEW}${IMG_KNOWN_GOOD}${IMG_OLD_KNOWN_GOOD}" > "${FAT_DIR}/${BOOTPART}"
}

# Read status variables
read_status_variables_bin() {

	export PARAMVALID="TRUE"
	
	STATALL=$(< "${FAT_DIR}/${BOOTPART}" )

	STAT_TRY_BOOT="${STATALL:0:1}"
	if [[ "$STAT_TRY_BOOT" =~ ^[0-9]$ ]]; then
		true
	else
		export PARAMVALID="FALSE"
	fi
	
	IMG_UNKNOWN_NEW="${STATALL:1:1}"
	if [[ "$IMG_UNKNOWN_NEW" =~ ^[0-9]$ ]]; then
		true
	else
		export PARAMVALID="FALSE"
	fi
	
	IMG_KNOWN_GOOD="${STATALL:2:1}"
	if [[ "$IMG_KNOWN_GOOD" =~ ^[0-9]$ ]]; then
		true
	else
		export PARAMVALID="FALSE"
	fi

	IMG_OLD_KNOWN_GOOD="${STATALL:3:1}"
	if [[ "$IMG_OLD_KNOWN_GOOD" =~ ^[0-9]$ ]]; then
		true
	else
		export PARAMVALID="FALSE"
	fi
	
	# If status info did not exist or was faulty, generate it now.
	if [ "$PARAMVALID" == "FALSE" ] ; then
		logecho "Warning: Invalid Status contents: '${STATALL}', creating default status"
		export STAT_TRY_BOOT='0'
		export IMG_UNKNOWN_NEW='0'
		export IMG_KNOWN_GOOD="${BOOTED_IMAGE}"
		export IMG_OLD_KNOWN_GOOD="${BOOTED_IMAGE}"
		write_status_variables_bin
	fi

}

print_status() {
	logecho "Current status: try_boot=${STAT_TRY_BOOT}, unknown_new=${IMG_UNKNOWN_NEW}, known_good=${IMG_KNOWN_GOOD} old_known_good=${IMG_OLD_KNOWN_GOOD}"
}

refresh_status() {

	logecho "Refreshing update status."

	# Get current status of update
	read_status_variables_bin
	print_status

	#
	# try_boot is always incremented by U-Boot.
	# Clear it to tell U-Boot that Linux was started sucessfully.
	#
	STAT_TRY_BOOT='0'
		
	# If we booted an unknown_new, it becomes our known_good now:
	if [ "${IMG_UNKNOWN_NEW}" == '1' ] ; then
		logecho "First succesful boot of freshly installed image 1"
		IMG_OLD_KNOWN_GOOD="${IMG_KNOWN_GOOD}"
		IMG_KNOWN_GOOD="${IMG_UNKNOWN_NEW}"
		IMG_UNKNOWN_NEW='0'
	elif [ "${IMG_UNKNOWN_NEW}" == '2' ] ; then
		logecho "First succesful boot of freshly installed image 2"
		IMG_OLD_KNOWN_GOOD="${IMG_KNOWN_GOOD}"
		IMG_KNOWN_GOOD="${IMG_UNKNOWN_NEW}"
		IMG_UNKNOWN_NEW='0'
	elif [ "${IMG_KNOWN_GOOD}" == '0' ] ; then
		# If known_good is 0 _and_ unknown_new is 0, we must have booted 
		# an old_known_good, which becomes the new known_good now.
		if [ "${IMG_OLD_KNOWN_GOOD}" == '1' ] ; then
			logecho "Booted old known good 1"
			IMG_KNOWN_GOOD="${IMG_OLD_KNOWN_GOOD}"
			IMG_OLD_KNOWN_GOOD='0'
		elif [ "${IMG_OLD_KNOWN_GOOD}" == '2' ] ; then
			logecho "Booted old known good 2"
			IMG_KNOWN_GOOD="${IMG_OLD_KNOWN_GOOD}"
			IMG_OLD_KNOWN_GOOD='0'
		fi
	fi
	
	#
	# At the very least, try_boot has changed...
	#
	write_status_variables_bin
	read_status_variables_bin
}
